/*
 * Manage the list of enclosures in the database
 */
#include <unistd.h>
#include <stdio.h>

#include "db.h"
#include "libfma.h"
#include "lf_dflt_error.h"
#include "lf_fabric.h"
#include "lf_fabric_db.h"

#define MAX_SWITCH 128

static void
usage()
{
  fprintf(stderr, "fm_switch is used to view and manage the list of enclosures in\n");
  fprintf(stderr, "the fabric.  This command should not be used to modify the list of\n");
  fprintf(stderr, "enclosures while fms is running - it is intended as a setup command\n");
  fprintf(stderr, "to be used before the fms is started.\n\n");
  fprintf(stderr, "Usage: fm_switch\n");
  fprintf(stderr, "  -a <switch_name> - add <switch_name> to the database of switches\n");
  fprintf(stderr, "  -d <switch_name> - remove  <switch_name> from the database of switches\n");
  fprintf(stderr, "  -h - print this help message\n");
  fprintf(stderr, "  -R <fms_run> - specify FMS_RUN directory\n");
  fprintf(stderr, "  -N <fms_db_name> - specify database name\n");
  fprintf(stderr, "  -V - print version\n");
  exit(1);
}

int
main(
  int argc,
  char **argv)
{
  int c;
  int i;
  int rc;
  extern char *optarg;
  char *db_name;
  char *fms_run;
  db_database_ptr_t dbp;
  struct lf_fabric *fp;
  struct lf_fabric_db *fdp;
  char *encnameadd[MAX_SWITCH];
  char *encnamedel[MAX_SWITCH];
  int ndel;
  int nadd;

  lf_init();

  /* check environment variables */
  fms_run = getenv(LF_ENV_FMS_RUN);
  if (fms_run == NULL || *fms_run == '\0') {
    fms_run = LF_DFLT_FMS_RUN;
  }
  db_name = getenv(LF_ENV_DB_NAME);
  if (db_name == NULL || *db_name == '\0') {
    db_name = LF_DFLT_DB_NAME;
  }

  nadd = 0;
  ndel = 0;

  /* command line args */
  while ((c = getopt(argc, argv, "hR:N:d:a:V")) != EOF) switch (c) {
  case 'h':
    usage();
    break;
  case 'R':
    fms_run = optarg;
    break;
  case 'N':
    db_name = optarg;
    break;
  case 'V':
    printf("FMS version is %s\n", Lf_version);
    exit(0);
    break;

  case 'a':
    if (nadd >= MAX_SWITCH) {
      fprintf(stderr, "Cannot add more than %d switches at once!\n",
	      MAX_SWITCH);
      exit(1);
    }
    encnameadd[nadd] = optarg;
    ++nadd;
    break;

  case 'd':
    if (ndel >= MAX_SWITCH) {
      fprintf(stderr, "Cannot delete more than %d switches at once!\n",
	      MAX_SWITCH);
      exit(1);
    }
    encnamedel[ndel] = optarg;
    ++ndel;
    break;
  }

  /* open the database */
  dbp = db_open_database(fms_run, db_name, 1);
  if (dbp == NULL) {
    LF_ERROR(("Error opening database, %s needs to exist and be writeable.",
	  fms_run));
  }

  fdp = lf_fabric_db_handle(dbp);
  if (fdp == NULL) LF_ERROR(("Error getting fabric db handle"));

  fp = lf_load_fabric(fdp);
  if (fp == NULL) {
    fprintf(stderr, "Error opening database at %s/%s\n", fms_run, db_name);
    exit(1);
  }

  /*
   * Add everything we have been asked to
   */
  if (nadd > 0) {
    for (i=0; i<nadd; ++i) {
      struct lf_enclosure *ep;

      /* first, check to see if this enclosure already exists */
      ep = lf_find_enclosure_by_name(fp, encnameadd[i]);

      /* If already present, skip adding it */
      if (ep != NULL) {
	fprintf(stderr, "%s already in database, skipping\n", encnameadd[i]);
	continue;
      }

      /* allocate the enclosure */
      LF_CALLOC(ep, struct lf_enclosure, 1);
      ep->name = encnameadd[i];
      ep->product_id = "";
      ep->db_status = LF_DS_CURRENT;

      /* Add this row to the database */
      rc = lf_add_enclosure_to_db(fdp, ep);
      if (rc != 0) {
	fprintf(stderr, "Error adding %s to database\n", encnameadd[i]);
	exit(1);
      }
      LF_FREE(ep);

      printf("Added %s to DB.\n", encnameadd[i]);
    }
    printf("\n");
  }

  /*
   * Delete everything we have been asked to.  Remove links, then linecards,
   * then enclosure itself so DB stays consistant in face of error.
   */
  if (ndel > 0) {
    for (i=0; i<ndel; ++i) {
      rc = lf_remove_link_from_db(fdp, encnamedel[i], -1, -1, -1);
      if (rc != 0) {
	fprintf(stderr, "Error removing %s's links from DB\n", encnamedel[i]);
	exit(1);
      }
      rc = lf_remove_linecard_from_db(fdp, encnamedel[i], -1);
      if (rc != 0) {
	fprintf(stderr, "Error removing %s's lincards from DB\n",
	        encnamedel[i]);
	exit(1);
      }
      rc = lf_remove_enclosure_from_db(fdp, encnamedel[i]);
      if (rc != 0) {
	fprintf(stderr, "Error removing %s from DB\n", encnamedel[i]);
	exit(1);
      }

      printf("Removed %s from DB.\n", encnamedel[i]);
    }

    printf("\n");
  }

  /*
   * If we did anything, re-load the database
   */
  if (nadd > 0 || ndel > 0) {

    /* flush the database to disk */
    rc = lf_flush_fabric_db(fdp);
    if (rc != 0) {
      fprintf(stderr, "Error writing DB back to disk\n");
      exit(1);
    }

    /* free the memory to be a good citizen */
    lf_free_fabric(fp);

    /* reload the fabric */
    fp = lf_load_fabric(fdp);
    if (fp == NULL) {
      fprintf(stderr, "Error re-loading fabric.\n");
      exit(1);
    }
  }

  /*
   * Dump the enclosure list
   */
  {
    int e;
    struct lf_enclosure *ep;

    if (fp->num_enclosures > 0) {

      printf("Enclosures in database:\n");
      for (e=0; e<fp->num_enclosures; ++e) {
	ep = fp->enclosures[e];
	printf("\t%s\n", ep->name);
      }
    } else {
      printf("No enclosures in database\n");
    }
  }

  exit(0);

 except:
  exit(1);
}
